
'use client';

import { useRestaurantInfo } from '@/hooks/use-restaurant-info';
import { useEffect, type ReactNode } from 'react';

// A component to safely inject HTML content from a string
const SafeHtmlInjector = ({ htmlContent }: { htmlContent: string }) => {
  // Creates a temporary div, injects the HTML, and then returns the children
  // This is a common pattern to parse an HTML string into React elements
  const createMarkup = (htmlString: string) => {
    if (typeof window === 'undefined') {
        return { __html: '' };
    }
    // Using DOMParser is safer than directly setting innerHTML on a persistent element
    const doc = new DOMParser().parseFromString(htmlString, 'text/html');
    return { __html: doc.body.innerHTML };
  };

  return <div dangerouslySetInnerHTML={createMarkup(htmlContent)} />;
};

// A component to inject scripts and other elements into the document head
const HeadInjector = ({ htmlContent }: { htmlContent: string }) => {
     useEffect(() => {
        if (!htmlContent || typeof window === 'undefined') return;

        const head = document.head;
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = htmlContent;

        const nodes = Array.from(tempDiv.childNodes);

        // Append nodes to the head. Use a fragment to do it in one operation.
        const fragment = document.createDocumentFragment();
        nodes.forEach(node => fragment.appendChild(node));
        head.appendChild(fragment);
        
        // Cleanup: remove the scripts when the component unmounts
        return () => {
            nodes.forEach(node => {
                if (head.contains(node)) {
                    head.removeChild(node);
                }
            });
        };
    }, [htmlContent]);

    return null; // This component doesn't render anything itself
}

export function TrackingProvider({ children }: { children: ReactNode }) {
  const { restaurantInfo, isLoaded } = useRestaurantInfo();

  return (
    <>
      {isLoaded && (
        <>
          {/* Inject scripts/tags into the <head> */}
          {restaurantInfo.facebookPixelHead && <HeadInjector htmlContent={restaurantInfo.facebookPixelHead} />}
          {restaurantInfo.googleTagManagerHead && <HeadInjector htmlContent={restaurantInfo.googleTagManagerHead} />}
          {restaurantInfo.googleAnalyticsHead && <HeadInjector htmlContent={restaurantInfo.googleAnalyticsHead} />}
          
          {/* Inject <noscript> or other tags at the beginning of <body> */}
          {restaurantInfo.facebookPixelBody && <SafeHtmlInjector htmlContent={restaurantInfo.facebookPixelBody} />}
          {restaurantInfo.googleTagManagerBody && <SafeHtmlInjector htmlContent={restaurantInfo.googleTagManagerBody} />}
          {restaurantInfo.googleAnalyticsBody && <SafeHtmlInjector htmlContent={restaurantInfo.googleAnalyticsBody} />}
        </>
      )}
      {children}
    </>
  );
}
